#include <arduino.h>
#include <EEPROM.h>
#include "zg_balancer.h"
#include "zg_capacity.h"

extern int gTmpTotalTimeLapsed;
extern char gPrintBuf[50];
uint8_t getSerialCmd(unsigned char *reqPayLoadBuf)
{
  static unsigned char rxState = RX_STATE_WAIT_FOR_START_BYTE;
  static unsigned char payLoadIndex = 0;
  static unsigned char payLoadLen = 0;

  unsigned char cmdReceived = 0;

  while (Serial.available() > 0)
  {
    switch (rxState)
    {
      case RX_STATE_WAIT_FOR_START_BYTE:
        if (START_BYTE == Serial.read())
        {
          rxState = RX_STATE_WAIT_FOR_LEN_BYTE;
        }
        break;
      case RX_STATE_WAIT_FOR_LEN_BYTE:
        payLoadLen = Serial.read();
        payLoadIndex = 0;
        rxState = RX_STATE_WAIT_FOR_PAYLOAD;
        break;
      case RX_STATE_WAIT_FOR_PAYLOAD:
        reqPayLoadBuf[payLoadIndex] = Serial.read();
        payLoadIndex++;

        if (payLoadIndex == payLoadLen)
        {
          if (1 == verifyChecksum(payLoadLen, reqPayLoadBuf))
          {
            cmdReceived = 1;
            rxState = RX_STATE_WAIT_FOR_START_BYTE;
          }
          else
          {
            Serial.println("Serial cmd rx checksum failed");
          }
        }
        break;
      default:
        Serial.println("Hit default case in rx state machine");
        break;
    }
  }
  return cmdReceived;
}

void checkHandleModeOverride(uint8_t cmdReceived, stSensorData *pstForceSensorData)
{
  //manage handle mode override
  //GUI tool has to keep sending override command within HANDLE_MODE_TIMEOUT_MS
  //else we abort override. This is for safety reasons
  static unsigned char handleModeState = HM_STATE_NONE;
  static unsigned long handleModeStartTime = 0;

  switch (handleModeState)
  {
    case HM_STATE_NONE:
      if (HANDLE_MODE_OVERRIDE_CMD == cmdReceived)
      {
        handleModeStartTime = millis();
        handleModeState = HM_STATE_TIME_CHECK;
      }
      break;
    case HM_STATE_TIME_CHECK:
      if (HANDLE_MODE_OVERRIDE_CMD == cmdReceived)
      {
        //restart timer
        handleModeStartTime = millis();
      }
      else if ((millis() - handleModeStartTime) > HANDLE_MODE_TIMEOUT_MS)
      {
        pstForceSensorData->trendManualOverride = HANDLE_MODE_OVERRIDE_CLEAR;
        handleModeState = HM_STATE_NONE;
      }
      break;
  }
  if (HANDLE_MODE_OVERRIDE_UP == pstForceSensorData->trendManualOverride)
  {
    pstForceSensorData->trendManualOverride = TREND_STATE_DECREASED;
  }
  else if (HANDLE_MODE_OVERRIDE_DOWN == pstForceSensorData->trendManualOverride)
  {
    pstForceSensorData->trendManualOverride = TREND_STATE_INCREASED;
  }
  else
  {
    //whatever decided in updateSensorData
  }
}

uint8_t processPayload(unsigned char *reqPayLoadBuf, stSensorData *pstForceSensorData,
                       stSensorData *pstControlSensorData, stZgSettings *pstZgSettings,
                       stZgData *pstZgData)
//uint8_t processPayload(unsigned char *reqPayLoadBuf)
{
  unsigned char looper = 0;
  unsigned long paramValue = 0;
  int paramId = 0;
  uint8_t cycleCountChanged = 0;

  unsigned char numParamIds2Read = 0;
  unsigned char numParamIds2Write = 0;
  unsigned char cmdByte = reqPayLoadBuf[0];
  unsigned char resBuf[MAX_PARAM_IDS * 5 + 6];

  unsigned char resPktLen = 0;
  int checksum = 0;
  unWord2Bytes word2ByteArr;

  switch (cmdByte)
  {
    case MULTIPLE_READ_CMD:
      numParamIds2Read = reqPayLoadBuf[1];

      resBuf[resPktLen++] = START_BYTE;
      //1 byte for response code, one byte for numParamIds and two byte checksum
      //each parameter id will add a len of 4 bytes for its value + 1 byte for its parameter id
      resBuf[resPktLen++] = (numParamIds2Read * 5) + 4;
      resBuf[resPktLen++] = MULTIPLE_READ_RESPONSE;
      resBuf[resPktLen++] = numParamIds2Read;

      for (looper = 0; looper < numParamIds2Read; looper++)
      {
        paramId = reqPayLoadBuf[looper + 2];

        switch (paramId)
        {
          case PARAM_ID_SW_VERSION:
            paramValue = SW_VERSION;
            break;
          case PARAM_ID_LOAD_CELL_DEAD_BAND_HI_LIMIT:
            paramValue = pstZgSettings->floatModeControlDeadbandHiLImit;
            break;
          case PARAM_ID_LOAD_CELL_DEAD_BAND_LO_LIMIT:
            paramValue = pstZgSettings->floatModeControlDeadbandLoLImit;
            break;
          case PARAM_ID_LIFT_SPEED:
            paramValue = pstZgSettings->liftSpeed;
            break;
          case PARAM_ID_LOWER_SPEED:
            paramValue = pstZgSettings->lowerSpeed;
            break;
          case PARAM_ID_INTERLOCK_WEIGHT:
            paramValue = pstZgSettings->interlockWeight;
            break;
          case PARAM_ID_MAX_WEIGHT:
            paramValue = pstZgSettings->maxWeightToLift;
            break;
          case PARAM_ID_MIN_WEIGHT:
            paramValue = pstZgSettings->noLoadBalancingWeight;
            break;
          case PARAM_ID_LOAD_CELL_SENSOR:
            paramValue = pstControlSensorData->rawAdcCounts;
            //paramValue = pstControlSensorData->filteredAdcCounts;
            break;
          case PARAM_ID_FORCE_SENSOR:
            paramValue = pstForceSensorData->rawAdcCounts;
            break;
          case PARAM_ID_PRESSURE_SENSOR:
            //paramValue = pstControlSensorData->rawAdcCounts;
            paramValue = adcRead(AI_PRESSURE_SENSOR);
            //paramValue = pstControlSensorData->restingValue;
            break;
          case PARAM_ID_SETTLING_TIME:
            //this is for engineering only
            paramValue = gTmpTotalTimeLapsed;
            //sprintf(gPrintBuf, "settling time = %d",gTmpTotalTimeLapsed);
            //Serial.println(gPrintBuf);
            break;
          case PARAM_ID_CURRENT_WEIGHT_RANGE:
            paramValue = getCurrentWeightRange(pstControlSensorData->rawAdcCounts);
            break;
          case PARAM_ID_CYCLE_COUNT:
            paramValue = pstZgData->cycleCount;
            break;
          case PARAM_ID_UN_USED37:
            paramValue = 0;
            break;
          default:
            resBuf[2] = NACK_RESPONSE;
            break;
        }
        // paramValue = getParamValue(paramId);
        // if(0xFFFFFFFF == paramValue)
        // {
        //   resBuf[2] = NACK_RESPONSE;
        // }
        resBuf[resPktLen++] = paramId;
        word2ByteArr.fourByteValue = 0;  //reset union
        word2ByteArr.fourByteValue = paramValue;
        resBuf[resPktLen++] = word2ByteArr.byteArr[0];
        resBuf[resPktLen++] = word2ByteArr.byteArr[1];
        resBuf[resPktLen++] = word2ByteArr.byteArr[2];
        resBuf[resPktLen++] = word2ByteArr.byteArr[3];
      }
      checksum = 0;
      for (looper = 2; looper < resPktLen; looper++)
      {
        checksum += resBuf[looper];
      }

      resBuf[resPktLen++] = checksum & 0xff;           //lsb
      resBuf[resPktLen++] = (checksum & 0xff00) >> 8;  //msb
      Serial.write(resBuf, resPktLen);
      break;
    case MULTIPLE_WRITE_CMD:
      numParamIds2Write = reqPayLoadBuf[1];

      resBuf[resPktLen++] = START_BYTE;
      //1 byte for response code, one byte for numParamIds and two byte checksum
      resBuf[resPktLen++] = (numParamIds2Write * 5) + 4;
      resBuf[resPktLen++] = MULTIPLE_WRITE_RESPONSE;
      resBuf[resPktLen++] = numParamIds2Write;

      for (looper = 2; looper <= numParamIds2Write * 5; looper += 5)
      {
        paramId = reqPayLoadBuf[looper];
        word2ByteArr.byteArr[0] = reqPayLoadBuf[looper + 1];
        word2ByteArr.byteArr[1] = reqPayLoadBuf[looper + 2];
        word2ByteArr.byteArr[2] = reqPayLoadBuf[looper + 3];
        word2ByteArr.byteArr[3] = reqPayLoadBuf[looper + 4];
        paramValue = word2ByteArr.fourByteValue;
        //response pkt is mostly echo of the req message
        resBuf[resPktLen++] = paramId;
        resBuf[resPktLen++] = word2ByteArr.byteArr[0];
        resBuf[resPktLen++] = word2ByteArr.byteArr[1];
        resBuf[resPktLen++] = word2ByteArr.byteArr[2];
        resBuf[resPktLen++] = word2ByteArr.byteArr[3];

        switch (paramId)
        {
          case PARAM_ID_LOAD_CELL_DEAD_BAND_HI_LIMIT:
            pstZgSettings->floatModeControlDeadbandHiLImit = paramValue;
            //update ram copy
            pstControlSensorData->controlDeadbandHiLImit = paramValue;
            pstControlSensorData->restingDeadbandHiLimit = paramValue;
            pstControlSensorData->restingValue = pstControlSensorData->filteredAdcCounts;
            break;
          case PARAM_ID_LOAD_CELL_DEAD_BAND_LO_LIMIT:
            pstZgSettings->floatModeControlDeadbandLoLImit = paramValue;
            //update ram copy
            pstControlSensorData->controlDeadbandLoLImit = paramValue;
            pstControlSensorData->restingDeadbandLoLimit = paramValue;
            pstControlSensorData->restingValue = pstControlSensorData->filteredAdcCounts;
            break;

          case PARAM_ID_LIFT_SPEED:
            pstZgSettings->liftSpeed = paramValue;
            break;
          case PARAM_ID_LOWER_SPEED:
            pstZgSettings->lowerSpeed = paramValue;
            break;
          case PARAM_ID_INTERLOCK_WEIGHT:
            //do not allow to clear the interlock unless load is on ground
            if ((0 == paramValue) && (pstZgSettings->interlockWeight > 0 ))
            {
              if (pstControlSensorData->rawAdcCounts < ONE_KG_WEIGHT)
              {
                pstZgSettings->interlockWeight = paramValue;
              }
              else
              {
                //nack the user
                resBuf[2] = NACK_RESPONSE;
              }
            }
            else
            {
              pstZgSettings->interlockWeight = paramValue;
            }
            break;
          case PARAM_ID_MAX_WEIGHT:
            if (paramValue > int(ZG_CAPACITY * 0.453592 * ONE_KG_WEIGHT * 0.8))
            {
              //nack the user
              resBuf[2] = NACK_RESPONSE;
            }
            else
            {
              pstZgSettings->maxWeightToLift = paramValue;
            }

            break;
          case PARAM_ID_MIN_WEIGHT:
            pstZgSettings->noLoadBalancingWeight = paramValue;
            break;

          case PARAM_ID_CYCLE_COUNT:
            pstZgData->cycleCount = paramValue;
            cycleCountChanged = 1;
            break;

          case PARAM_ID_UN_USED37:
            //gstUserSpeedSetting.lowerFine = paramValue;
            break;
          default:
            resBuf[2] = NACK_RESPONSE;
            break;
        }
        //  if(0 == setParamValue(paramId, paramValue))
        //  {
        //    resBuf[2] = NACK_RESPONSE;
        //  }
      }
      checksum = 0;
      for (looper = 2; looper < resPktLen; looper++)
      {
        checksum += resBuf[looper];
      }
      resBuf[resPktLen++] = checksum & 0xff;           //lsb
      resBuf[resPktLen++] = (checksum & 0xff00) >> 8;  //msb
      if (resBuf[2] |= NACK_RESPONSE)
      {

        WriteToEeprom(EEPROM_ADDR_SETTINGS_START, (uint8_t *)pstZgSettings, sizeof(stZgSettings));
        if (1 == cycleCountChanged)
        {
          WriteToEeprom(EEPROM_ADDR_ZG_DATA_START, (uint8_t *)pstZgData, sizeof(stZgData));
          cycleCountChanged = 0;
        }
      }
      Serial.write(resBuf, resPktLen);
      break;
    case KEEP_ALIVE_CMD:
      resBuf[resPktLen++] = START_BYTE;
      resBuf[resPktLen++] = PING_RESPONSE_LEN;  //Length excludes start byte
      resBuf[resPktLen++] = KEEP_ALIVE_RESPONSE;

      //echo packet has data in the form of 0xaa, 0x55, 0xaa, 0x55 ...
      for (looper = 0; looper < NUM_PING_BYTES; looper += 2)
      {
        resBuf[resPktLen++] = 0xAA;
        resBuf[resPktLen++] = 0x55;
      }

      checksum = 0;
      for (looper = 2; looper < resPktLen; looper++)
      {
        checksum += resBuf[looper];
      }
      resBuf[resPktLen++] = checksum & 0xff;           //lsb
      resBuf[resPktLen++] = (checksum & 0xff00) >> 8;  //msb
      Serial.write(resBuf, resPktLen);
      break;
    case SET_SERIAL_NUM_CMD:
        memcpy(pstZgData->serialNumber, &reqPayLoadBuf[1], SERIAL_NUMBER_LEN);
        WriteToEeprom(EEPROM_ADDR_ZG_DATA_START, (uint8_t *)pstZgData, sizeof(stZgData));
        
        resBuf[resPktLen++] = START_BYTE;
        resBuf[resPktLen++] = SET_SERIAL_NUMBER_RESPONSE_LEN;  //Length excludes start byte
        resBuf[resPktLen++] = SET_SERIAL_NUM_RESPONSE;

        for (looper = 0; looper < SERIAL_NUMBER_LEN; looper++)
        {
          resBuf[resPktLen++] = reqPayLoadBuf[1 + looper];
        }

        checksum = 0;
        for (looper = 2; looper < resPktLen; looper++)
        {
          checksum += resBuf[looper];
        }
        resBuf[resPktLen++] = checksum & 0xff;           //lsb
        resBuf[resPktLen++] = (checksum & 0xff00) >> 8;  //msb
        Serial.write(resBuf, resPktLen);

      break;
    case GET_SERIAL_NUM_CMD:
        resBuf[resPktLen++] = START_BYTE;
        resBuf[resPktLen++] = GET_SERIAL_NUMBER_RESPONSE_LEN;  //Length excludes start byte
        resBuf[resPktLen++] = GET_SERIAL_NUM_RESPONSE;

        for (looper = 0; looper < SERIAL_NUMBER_LEN; looper++)
        {
          resBuf[resPktLen++] = pstZgData->serialNumber[looper];
        }

        checksum = 0;
        for (looper = 2; looper < resPktLen; looper++)
        {
          checksum += resBuf[looper];
        }
        resBuf[resPktLen++] = checksum & 0xff;           //lsb
        resBuf[resPktLen++] = (checksum & 0xff00) >> 8;  //msb
        Serial.write(resBuf, resPktLen);

      break;

    case HANDLE_MODE_OVERRIDE_CMD:
      resBuf[resPktLen++] = START_BYTE;
      resBuf[resPktLen++] = HANDLE_MODE_RESPONSE_LEN;  //Length excludes start byte

      if ((HANDLE_MODE_OVERRIDE_CLEAR == reqPayLoadBuf[1]) || (HANDLE_MODE_OVERRIDE_UP == reqPayLoadBuf[1]) || (HANDLE_MODE_OVERRIDE_DOWN == reqPayLoadBuf[1]))
      {
        pstForceSensorData->trendManualOverride = reqPayLoadBuf[1];
        resBuf[resPktLen++] = HANDLE_MODE_OVERRIDE_RESPONSE;
      }
      else
      {
        pstForceSensorData->trendManualOverride = HANDLE_MODE_OVERRIDE_CLEAR;
        resBuf[resPktLen++] = NACK_RESPONSE;
      }
      resBuf[resPktLen++] = reqPayLoadBuf[1];
      checksum = 0;
      for (looper = 2; looper < resPktLen; looper++)
      {
        checksum += resBuf[looper];
      }
      resBuf[resPktLen++] = checksum & 0xff;           //lsb
      resBuf[resPktLen++] = (checksum & 0xff00) >> 8;  //msb
      Serial.write(resBuf, resPktLen);

      break;
    default:
      Serial.println("Unsupported cmd received");
      break;
  }
  return cmdByte;
}

int verifyChecksum(int payLoadLen, unsigned char *payLoadBuf)
{
  //payload len is inclusive of checksum
  unsigned char msb = payLoadBuf[payLoadLen - 1];
  unsigned char lsb = payLoadBuf[payLoadLen - 2];

  int checkSumReceived = (msb << 8) | lsb;

  unsigned char looper = 0;
  int checksumActual = 0;

  for (looper = 0; looper < (payLoadLen - 2); looper++)
  {
    checksumActual += payLoadBuf[looper];
  }

  if (checksumActual == checkSumReceived)
  {
    return 1;
  }
  else
  {
    return 0;
  }
}
