#include <arduino.h>
#include <EEPROM.h>
#include "zg_balancer.h"
#include "zg_capacity.h"

char gPrintBuf[50];

static stZgSettings gstZgSettings;
static stZgData gstZgData;
static stSensorData gstForceSensorData;
static stSensorData gstControlSensorData;

stWtRangeVsSpeedLookUp gstWtVsSpeedLow[MAX_SPEED_RANGES];
stWtRangeVsSpeedLookUp gstWtVsSpeedMed[MAX_SPEED_RANGES];
stWtRangeVsSpeedLookUp gstWtVsSpeedHi[MAX_SPEED_RANGES];

static uint8_t reqPayLoadBuf[50];

void setup()
{
  //reset settings. Remove later
  //memset(gPrintBuf, 0xFF, sizeof(gPrintBuf));
  //WriteToEeprom(EEPROM_ADDR_START, gPrintBuf, sizeof(gPrintBuf));

  Serial.begin(9600);
  initLedOutputs();
  initInterlockInputs();
  initExhaustSolenoids();
  initIntakeSolenoids();
  initSettings(&gstZgSettings);
  initZgData(&gstZgData);
  createWeightVsSpeedLookUpTable();
  digitalWrite(LED_ERROR, HIGH);  //indicate we are busy
  initSensorData(&gstForceSensorData, AI_FORCE_SENSOR, &gstZgSettings);

  initSensorData(&gstControlSensorData, AI_CONTROL_SENSOR, &gstZgSettings);

  digitalWrite(LED_ERROR, LOW);
  //sprintf(gPrintBuf, "Force sensor resting resting value = %d", gstForceSensorData.restingValue);
  //Serial.println(gPrintBuf);
  //sprintf(gPrintBuf, "Load cell resting resting value = %d", gstControlSensorData.restingValue);
  //Serial.println(gPrintBuf);
}

void loop()
{
  // put your main code here, to run repeatedly:
  uint8_t currentHandleModeState;
  uint8_t currentFloatModeState;
  uint8_t serialCmdReceived = 0;
  uint16_t rawAdcCountControlSensor;
  uint16_t rawAdcCountForceSensor;
  uint8_t sensorCheckState;
  while (1)
  {
    rawAdcCountForceSensor   = adcRead(AI_FORCE_SENSOR);
    rawAdcCountControlSensor = adcRead(AI_CONTROL_SENSOR);

    sensorCheckState = checkSensors(rawAdcCountControlSensor, rawAdcCountForceSensor);

    //checkHandleModeOverride(serialCmdReceived, &gstForceSensorData);
    if(SENSOR_CHECK_STATE_NORMAL == sensorCheckState)
    {
      updateSensorData(&gstForceSensorData, rawAdcCountForceSensor);
      updateSensorData(&gstControlSensorData, rawAdcCountControlSensor);

      //handle mode and float mode are mutually exclusive
      currentHandleModeState = runHandleMode(&gstForceSensorData, &gstControlSensorData, &gstZgSettings);

      if (HANDLE_MODE_STATE_IDLE == currentHandleModeState)
      {
        currentFloatModeState = runFloatMode(&gstControlSensorData, &gstZgSettings);
      }
      
      handleCycleCount(&gstControlSensorData, &gstZgData, &gstZgSettings);
      handleInterlock(&gstControlSensorData, &gstZgSettings);
    }

    serialCmdReceived = 0;
    if (1 == getSerialCmd(reqPayLoadBuf))
    {
      serialCmdReceived = processPayload(reqPayLoadBuf, &gstForceSensorData, &gstControlSensorData, &gstZgSettings, &gstZgData);
      //serialCmdReceived = processPayload(reqPayLoadBuf);
    }

  }
}

void startPressureDecrease(int airExhaustSpeed)
{
  switch (airExhaustSpeed)
  {
    case 0:
      digitalWrite(EXHAUST_LOW_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(EXHAUST_MED_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(EXHAUST_HIGH_SOLENOID, SOLENOID_CLOSE);
      break;
    case 1:
      digitalWrite(EXHAUST_LOW_SOLENOID, SOLENOID_OPEN);
      digitalWrite(EXHAUST_MED_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(EXHAUST_HIGH_SOLENOID, SOLENOID_CLOSE);
      break;
    case 2:
      digitalWrite(EXHAUST_LOW_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(EXHAUST_MED_SOLENOID, SOLENOID_OPEN);
      digitalWrite(EXHAUST_HIGH_SOLENOID, SOLENOID_CLOSE);
      break;
    case 3:
      digitalWrite(EXHAUST_LOW_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(EXHAUST_MED_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(EXHAUST_HIGH_SOLENOID, SOLENOID_OPEN);
      break;
    case 4:
      digitalWrite(EXHAUST_LOW_SOLENOID, SOLENOID_OPEN);
      digitalWrite(EXHAUST_MED_SOLENOID, SOLENOID_OPEN);
      digitalWrite(EXHAUST_HIGH_SOLENOID, SOLENOID_CLOSE);
      break;
    case 5:
      digitalWrite(EXHAUST_LOW_SOLENOID, SOLENOID_OPEN);
      digitalWrite(EXHAUST_MED_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(EXHAUST_HIGH_SOLENOID, SOLENOID_OPEN);
      break;
    case 6:
      digitalWrite(EXHAUST_LOW_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(EXHAUST_MED_SOLENOID, SOLENOID_OPEN);
      digitalWrite(EXHAUST_HIGH_SOLENOID, SOLENOID_OPEN);
      break;
    case 7:
      digitalWrite(EXHAUST_LOW_SOLENOID, SOLENOID_OPEN);
      digitalWrite(EXHAUST_MED_SOLENOID, SOLENOID_OPEN);
      digitalWrite(EXHAUST_HIGH_SOLENOID, SOLENOID_OPEN);
      break;
    default:
      Serial.println("Hit default case in airExhaustSpeed");
      break;
  }
}

void airIntake(uint8_t speedSetting, uint8_t weightRange)
{
  uint8_t valveCombinationLifting;
  uint8_t valveCombinationLowering;
  switch (speedSetting)
  {
    case LOW_SPEED:
      valveCombinationLifting = gstWtVsSpeedLow[weightRange].mValveLiftingCombination;
      valveCombinationLowering = gstWtVsSpeedLow[weightRange].dValveLiftingCombination;
      break;
    case MED_SPEED:
      valveCombinationLifting = gstWtVsSpeedMed[weightRange].mValveLiftingCombination;
      valveCombinationLowering = gstWtVsSpeedMed[weightRange].dValveLiftingCombination;
      break;
    case HIGH_SPEED:
      valveCombinationLifting = gstWtVsSpeedHi[weightRange].mValveLiftingCombination;
      valveCombinationLowering = gstWtVsSpeedHi[weightRange].dValveLiftingCombination;
      break;
    default:
      //shutoff intake and exhaust
      valveCombinationLifting = 0;
      valveCombinationLowering = 0;
      break;
  }

  startPressureIncrease(valveCombinationLifting);
  startPressureDecrease(valveCombinationLowering);
}

void airExhaust(uint8_t speedSetting, uint8_t weightRange)
{
  uint8_t valveCombinationLifting;
  uint8_t valveCombinationLowering;
  switch (speedSetting)
  {
    case LOW_SPEED:
      valveCombinationLifting = gstWtVsSpeedLow[weightRange].mValveLoweringCombination;
      valveCombinationLowering = gstWtVsSpeedLow[weightRange].dValveLoweringCombination;
      break;
    case MED_SPEED:
      valveCombinationLifting = gstWtVsSpeedMed[weightRange].mValveLoweringCombination;
      valveCombinationLowering = gstWtVsSpeedMed[weightRange].dValveLoweringCombination;
      break;
    case HIGH_SPEED:
      valveCombinationLifting = gstWtVsSpeedHi[weightRange].mValveLoweringCombination;
      valveCombinationLowering = gstWtVsSpeedHi[weightRange].dValveLoweringCombination;
      break;
    default:
      //shutoff intake and exhaust
      valveCombinationLifting = 0;
      valveCombinationLowering = 0;
      break;
  }

  startPressureIncrease(valveCombinationLifting);
  startPressureDecrease(valveCombinationLowering);
}

void startPressureIncrease(int airIntakeSpeed)
{
  switch (airIntakeSpeed)
  {
    case 0:
      digitalWrite(INTAKE_LOW_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(INTAKE_MED_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(INTAKE_HIGH_SOLENOID, SOLENOID_CLOSE);
      break;
    case 1:
      digitalWrite(INTAKE_LOW_SOLENOID, SOLENOID_OPEN);
      digitalWrite(INTAKE_MED_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(INTAKE_HIGH_SOLENOID, SOLENOID_CLOSE);
      break;

    case 2:
      digitalWrite(INTAKE_LOW_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(INTAKE_MED_SOLENOID, SOLENOID_OPEN);
      digitalWrite(INTAKE_HIGH_SOLENOID, SOLENOID_CLOSE);
      break;
    case 3:
      digitalWrite(INTAKE_LOW_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(INTAKE_MED_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(INTAKE_HIGH_SOLENOID, SOLENOID_OPEN);
      break;
    case 4:
      digitalWrite(INTAKE_LOW_SOLENOID, SOLENOID_OPEN);
      digitalWrite(INTAKE_MED_SOLENOID, SOLENOID_OPEN);
      digitalWrite(INTAKE_HIGH_SOLENOID, SOLENOID_CLOSE);
      break;
    case 5:
      digitalWrite(INTAKE_LOW_SOLENOID, SOLENOID_OPEN);
      digitalWrite(INTAKE_MED_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(INTAKE_HIGH_SOLENOID, SOLENOID_OPEN);
      break;
    case 6:
      digitalWrite(INTAKE_LOW_SOLENOID, SOLENOID_CLOSE);
      digitalWrite(INTAKE_MED_SOLENOID, SOLENOID_OPEN);
      digitalWrite(INTAKE_HIGH_SOLENOID, SOLENOID_OPEN);
      break;
    case 7:
      digitalWrite(INTAKE_LOW_SOLENOID, SOLENOID_OPEN);
      digitalWrite(INTAKE_MED_SOLENOID, SOLENOID_OPEN);
      digitalWrite(INTAKE_HIGH_SOLENOID, SOLENOID_OPEN);
      break;
    default:
      Serial.println("Hit default case in airIntakeSpeed");
      break;
  }
}

void stopAirFlow()
{
  //exhaust air using low diameter solenoid
  digitalWrite(EXHAUST_LOW_SOLENOID, SOLENOID_CLOSE);
  digitalWrite(EXHAUST_MED_SOLENOID, SOLENOID_CLOSE);
  digitalWrite(EXHAUST_HIGH_SOLENOID, SOLENOID_CLOSE);

  //exhaust air using low diameter solenoid
  digitalWrite(INTAKE_LOW_SOLENOID, SOLENOID_CLOSE);
  digitalWrite(INTAKE_MED_SOLENOID, SOLENOID_CLOSE);
  digitalWrite(INTAKE_HIGH_SOLENOID, SOLENOID_CLOSE);
}

void initIntakeSolenoids()
{
  //Initialize intake solenoid valves
  pinMode(INTAKE_LOW_SOLENOID, OUTPUT);
  digitalWrite(INTAKE_LOW_SOLENOID, LOW);

  pinMode(INTAKE_MED_SOLENOID, OUTPUT);
  digitalWrite(INTAKE_MED_SOLENOID, LOW);

  pinMode(INTAKE_HIGH_SOLENOID, OUTPUT);
  digitalWrite(INTAKE_HIGH_SOLENOID, LOW);
}

void initExhaustSolenoids()
{
  //Initialize exhaust solenoid valves
  pinMode(EXHAUST_LOW_SOLENOID, OUTPUT);
  digitalWrite(EXHAUST_LOW_SOLENOID, LOW);

  pinMode(EXHAUST_MED_SOLENOID, OUTPUT);
  digitalWrite(EXHAUST_MED_SOLENOID, LOW);

  pinMode(EXHAUST_HIGH_SOLENOID, OUTPUT);
  digitalWrite(EXHAUST_HIGH_SOLENOID, LOW);
}

void initLedOutputs()
{
  pinMode(LED1, OUTPUT);
  digitalWrite(LED1, LOW);

  pinMode(LED2, OUTPUT);
  digitalWrite(LED2, LOW);

  pinMode(LED3, OUTPUT);
  digitalWrite(LED3, LOW);

  pinMode(LED_ERROR, OUTPUT);
  digitalWrite(LED_ERROR, LOW);
}

void initInterlockInputs()
{
  //Initialize digital inputs
  pinMode(DI_ENGAGE_CLAMPING, INPUT);
  digitalWrite(DI_ENGAGE_CLAMPING, HIGH);
  pinMode(DI_DISENGAGE_CLAMPING, INPUT);
  digitalWrite(DI_DISENGAGE_CLAMPING, HIGH);
}

int adcRead(int ch)
{
  //dummy read as per arduino community
  analogRead(ch);
  analogRead(ch);
  //actual read
  return analogRead(ch);
}

void initSettings(stZgSettings *pStZgSettings)
{
  uint8_t zgSettings[sizeof(stZgSettings)];
  int32_t eepromAddr = EEPROM_ADDR_SETTINGS_START;

  //load previously stored speed settings from EEPROM
  for (uint8_t byteNo = 0; byteNo < sizeof(stZgSettings); byteNo++)
  {
    zgSettings[byteNo] = EEPROM.read(eepromAddr++);
  }

  memcpy((uint8_t *)pStZgSettings, zgSettings, sizeof(stZgSettings));

  switch (pStZgSettings->liftSpeed)
  {
    case LOW_SPEED:
    case MED_SPEED:
    case HIGH_SPEED:
      //ok then
      break;
    default:
      pStZgSettings->liftSpeed = MED_SPEED;
      break;
  }

  switch (pStZgSettings->lowerSpeed)
  {
    case LOW_SPEED:
    case MED_SPEED:
    case HIGH_SPEED:
      //ok good
      break;
    default:
      pStZgSettings->lowerSpeed = MED_SPEED;
      break;
  }

  if (0xFFFF == pStZgSettings->interlockWeight)
  {
    pStZgSettings->interlockWeight = 0;
  }
  if (0xFFFF == pStZgSettings->noLoadBalancingWeight)
  {
    pStZgSettings->noLoadBalancingWeight = 0;
  }
  if (0xFFFF == pStZgSettings->maxWeightToLift)
  {
    //convert pounds to kg to adc counts and limit the capacity to 80%
    pStZgSettings->maxWeightToLift = int(ZG_CAPACITY * 0.453592 * ONE_KG_WEIGHT * 0.8); 
  }
  
  //control sensor/load cell deadbands
  if (0xFF == pStZgSettings->floatModeControlDeadbandHiLImit)
  {
    pStZgSettings->floatModeControlDeadbandHiLImit = CONTROL_SENSOR_CONTROL_DEAD_BAND_HI_LIMIT;
  }

  if (0xFF == pStZgSettings->floatModeControlDeadbandLoLImit)
  {
    pStZgSettings->floatModeControlDeadbandLoLImit = CONTROL_SENSOR_CONTROL_DEAD_BAND_LO_LIMIT;
  }

  if (0xFF == pStZgSettings->floatModeRestingDeadbandHiLImit)
  {
    pStZgSettings->floatModeRestingDeadbandHiLImit = CONTROL_SENSOR_RESTING_DEAD_BAND_HI_LIMIT;
  }

  if (0xFF == pStZgSettings->floatModeRestingDeadbandLoLImit)
  {
    pStZgSettings->floatModeRestingDeadbandLoLImit = CONTROL_SENSOR_RESTING_DEAD_BAND_LO_LIMIT;
  }

  //handle sensor deadbands
  if (0xFF == pStZgSettings->handleModeControlDeadbandHiLImit)
  {
    pStZgSettings->handleModeControlDeadbandHiLImit = FORCE_SENSOR_CONTROL_DEAD_BAND_HI_LIMIT;
  }

  if (0xFF == pStZgSettings->handleModeControlDeadbandLoLImit)
  {
    pStZgSettings->handleModeControlDeadbandLoLImit = FORCE_SENSOR_CONTROL_DEAD_BAND_LO_LIMIT;
  }

  if (0xFF == pStZgSettings->handleModeRestingDeadbandHiLImit)
  {
    pStZgSettings->handleModeRestingDeadbandHiLImit = FORCE_SENSOR_RESTING_DEAD_BAND_HI_LIMIT;
  }

  if (0xFF == pStZgSettings->handleModeRestingDeadbandLoLImit)
  {
    pStZgSettings->handleModeRestingDeadbandLoLImit = FORCE_SENSOR_RESTING_DEAD_BAND_LO_LIMIT;
  }
}

void initZgData(stZgData *pStZgData)
{
  uint8_t zgData[NUM_ZG_DATA_BYTES];
  int32_t eepromAddr = EEPROM_ADDR_ZG_DATA_START;
  for (uint8_t byteNo = 0; byteNo < sizeof(stZgData); byteNo++)
  {
    zgData[byteNo] = EEPROM.read(eepromAddr++);
  }

  memcpy((uint8_t *)pStZgData, zgData, sizeof(stZgData));
  if (0xFFFFFFFF == pStZgData->cycleCount)
  {
    pStZgData->cycleCount = 0;
  }
}

void WriteToEeprom(int32_t eepromAddr, uint8_t *dataBuf, int16_t numBytesToWrite)
{
  for (uint8_t byteNo = 0; byteNo < numBytesToWrite; byteNo++)
  {
    EEPROM.write(eepromAddr++, dataBuf[byteNo]);
  }
}

void initSensorData(stSensorData *pstSensorData, uint8_t ch, stZgSettings *pstZgSettings)
{
  unsigned long timerStart;
  float newFilteredReading;
  float oldFilteredReading = adcRead(ch);
  uint16_t adcCount;

  memset(pstSensorData, 0, sizeof(stSensorData));

  pinMode(ch, INPUT_PULLUP);
  //for one sec, calculate the resting value. We are not checking if sensor
  //is really resting. Simply assuming. TODO find the actual resting value
  timerStart = millis();
  while ((millis() - timerStart) < 1000)
  {
    adcCount = adcRead(ch);
    newFilteredReading = (oldFilteredReading * 0.9) + (adcCount * 0.1);
    pstSensorData->restingValue = int(newFilteredReading + 0.5);
    oldFilteredReading = newFilteredReading;
  }

  if (AI_CONTROL_SENSOR == ch)
  {
    pstSensorData->isControlSensor = 1;
    pstSensorData->controlDeadbandHiLImit = pstZgSettings->floatModeControlDeadbandHiLImit;
    pstSensorData->controlDeadbandLoLImit = pstZgSettings->floatModeControlDeadbandLoLImit;
    pstSensorData->restingDeadbandHiLimit = pstZgSettings->floatModeRestingDeadbandHiLImit;
    pstSensorData->restingDeadbandLoLimit = pstZgSettings->floatModeRestingDeadbandLoLImit;
    //adjust for tare value
    if (pstSensorData->restingValue < NO_LOAD_WEIGHT_ADC_COUNT)
    {
      pstSensorData->restingValue = 0;
    }
    else
    {
      pstSensorData->restingValue = pstSensorData->restingValue - NO_LOAD_WEIGHT_ADC_COUNT;
    }
  }
  else
  {
    pstSensorData->isControlSensor = 0;
    pstSensorData->controlDeadbandHiLImit = pstZgSettings->handleModeControlDeadbandHiLImit;
    pstSensorData->controlDeadbandLoLImit = pstZgSettings->handleModeControlDeadbandLoLImit;
    pstSensorData->restingDeadbandHiLimit = pstZgSettings->handleModeRestingDeadbandHiLImit;
    pstSensorData->restingDeadbandLoLimit = pstZgSettings->handleModeRestingDeadbandLoLImit;
  }
}

void updateSensorData(stSensorData *pstSensorData, uint16_t currentAdcCounts)
{
  float newFilteredReading;
  //is overridden externally from GUI tool?
  if (HANDLE_MODE_OVERRIDE_UP == pstSensorData->trendManualOverride)
  {
    pstSensorData->trend = TREND_STATE_DECREASED;
    return;
  }
  else if (HANDLE_MODE_OVERRIDE_DOWN == pstSensorData->trendManualOverride)
  {
    pstSensorData->trend = TREND_STATE_INCREASED;
    return;
  }
  else
  {
    //find latest trend in below code
  }

  if (1 == pstSensorData->isControlSensor)
  {
    //adjust for load cell and hanlde weight offset
    if (currentAdcCounts < NO_LOAD_WEIGHT_ADC_COUNT)
    {
      pstSensorData->rawAdcCounts = 0;
    }
    else
    {
      pstSensorData->rawAdcCounts = currentAdcCounts - NO_LOAD_WEIGHT_ADC_COUNT;
    }
  }
  else
  {
    pstSensorData->rawAdcCounts = currentAdcCounts;
  }

  pstSensorData->delta = pstSensorData->rawAdcCounts - pstSensorData->restingValue;

  if (pstSensorData->delta > pstSensorData->controlDeadbandHiLImit)
  {
    pstSensorData->trend = TREND_STATE_INCREASED;
  }
  else if (pstSensorData->delta < -pstSensorData->controlDeadbandLoLImit)
  {
    pstSensorData->trend = TREND_STATE_DECREASED;
  }
  else if (pstSensorData->delta > pstSensorData->restingDeadbandHiLimit)
  {
    //stay in prev state. This could be considered hysteresis?
  }
  else if (pstSensorData->delta < -pstSensorData->restingDeadbandLoLimit)
  {
    //stay in prev state. This could be considered hysteresis
    //else we will be oscillating around control deadband
  }
  else
  {
    pstSensorData->trend = TREND_STATE_RESTING;
  }

  //first time, there will be no old reading
  if (0 == pstSensorData->oldFilteredReading)
  {
    pstSensorData->oldFilteredReading = pstSensorData->rawAdcCounts;
  }

  newFilteredReading = (pstSensorData->oldFilteredReading * 0.9) + (pstSensorData->rawAdcCounts * 0.1);
  pstSensorData->filteredAdcCounts = int(newFilteredReading + 0.5);

  pstSensorData->oldFilteredReading = newFilteredReading;
}

// unsigned long getParamValue(uint8_t paramId)
// {
//   switch (paramId)
//   {
//     case PARAM_ID_SW_VERSION:
//       return SW_VERSION;
//     break;
//     case PARAM_ID_LOAD_CELL_DEAD_BAND:
//       return gstZgSettings.controlDeadband;
//     break;
//     case PARAM_ID_LIFT_SPEED:
//       return gstZgSettings.liftSpeed;
//     break;
//     case PARAM_ID_LOWER_SPEED:
//       return gstZgSettings.lowerSpeed;
//     break;
//     case PARAM_ID_TARE_VALUE:
//       return gstZgSettings.tareSetting;
//     break;
//     case PARAM_ID_UN_USED6:
//       //paramValue = gstUserSpeedSetting.lowerCoarse;
//       return 0;
//     break;
//     case PARAM_ID_UN_USED7:
//       //paramValue = gstUserSpeedSetting.lowerFine;
//       return 0;
//     break;
//     case PARAM_ID_LOAD_CELL_SENSOR:
//       return gstControlSensorData.rawAdcCounts;
//       //paramValue = pstControlSensorData->filteredAdcCounts;
//     break;
//     case PARAM_ID_FORCE_SENSOR:
//       return gstForceSensorData.rawAdcCounts;
//     break;
//     case PARAM_ID_PRESSURE_SENSOR:
//       //paramValue = pstControlSensorData->rawAdcCounts;
//       return adcRead(AI_PRESSURE_SENSOR);
//     break;
//     case PARAM_ID_SETTLING_TIME:
//       //this is for engineering only
//       //return gTmpTotalTimeLapsed;
//     break;
//     case PARAM_ID_CURRENT_WEIGHT_RANGE:
//       return getCurrentWeightRange(gstControlSensorData.rawAdcCounts);
//     break;
//     case PARAM_ID_CYCLE_COUNT:
//       return gstZgData.cycleCount;
//     break;
//     case PARAM_ID_UN_USED37:
//       return  0;
//     break;
//     default:
//       //resBuf[2] = NACK_RESPONSE;
//       Serial.println("\nHit default case in getParamValue");
//       return 0xFFFFFFFF;
//     break;
//   }
// }

// uint8_t setParamValue(uint8_t paramId, unsigned long paramValue)
// {
//     uint8_t setStatus = 1;

//     switch (paramId)
//     {
//       case PARAM_ID_LOAD_CELL_DEAD_BAND:
//         gstZgSettings.controlDeadband = paramValue;
//         gstControlSensorData.controlDeadband = paramValue;
//       break;
//       case PARAM_ID_LIFT_SPEED:
//           gstZgSettings.liftSpeed = paramValue;
//       break;
//       case PARAM_ID_LOWER_SPEED:
//         gstZgSettings.lowerSpeed = paramValue;
//       break;
//       case PARAM_ID_TARE_VALUE:
//         gstZgSettings.tareSetting = paramValue;
//         gstControlSensorData.tareSettingChanged = 1;
//       break;
//       case PARAM_ID_CYCLE_COUNT:
//         gstZgData.cycleCount = paramValue;
//         //cycleCountChanged = 1;
//       break;
//       case PARAM_ID_UN_USED37:
//         //gstUserSpeedSetting.lowerFine = paramValue;
//       break;
//       default:
//         //resBuf[2] = NACK_RESPONSE;
//         setStatus = 0;
//       break;
//     }
//   return setStatus;
// }

uint8_t checkSensors(uint16_t controlSensorAdcCount, uint16_t forceSensorAdcCount)
{
    static uint8_t sensorCheckState = SENSOR_CHECK_STATE_CHECK_RESTING_VALUES;
    static unsigned long timerStartValue = 0;
    static uint8_t ledToggle = 1;
    
    switch(sensorCheckState)
    {
        case SENSOR_CHECK_STATE_CHECK_RESTING_VALUES:
          sprintf(gPrintBuf, "FS resting resting value = %d", forceSensorAdcCount);
          Serial.println(gPrintBuf);
          sprintf(gPrintBuf, "CS resting resting value = %d", controlSensorAdcCount);
          Serial.println(gPrintBuf);
          if(controlSensorAdcCount < LOAD_CELL_INPUT_SHORTED_VALUE)
          {
            Serial.println("\nLoad cell input too low. May be shorted!");
            sensorCheckState = SENSOR_CHECK_STATE_OUT_OF_RANGE;
          }
          else if(controlSensorAdcCount > LOAD_CELL_INPUT_OPEN_VALUE)
          {
            Serial.println("\nLoad cell input too high. May be open!");
            sensorCheckState = SENSOR_CHECK_STATE_OUT_OF_RANGE;
          }
          else if(forceSensorAdcCount < FORCE_SENSOR_INPUT_SHORTED_VALUE)
          {
            Serial.println("\nForce sensor input too low. May be shorted!");
            sensorCheckState = SENSOR_CHECK_STATE_OUT_OF_RANGE;
          }
          else if(forceSensorAdcCount > FORCE_SENSOR_INPUT_OPEN_VALUE)
          {
            Serial.println("\nForce sensor input too high. May be open!");
            sensorCheckState = SENSOR_CHECK_STATE_OUT_OF_RANGE;
          }
          else if(forceSensorAdcCount < FORCE_SENSOR_INPUT_RESTING_LOW_LIMIT)
          {
            Serial.println("\nForce sensor value < resting value");
            sensorCheckState = SENSOR_CHECK_STATE_OUT_OF_RANGE;
          }
          else if(forceSensorAdcCount > FORCE_SENSOR_INPUT_RESTING_HIGH_LIMIT)
          {
            Serial.println("\nForce sensor value > resting value");
            sensorCheckState = SENSOR_CHECK_STATE_OUT_OF_RANGE;
          }
          else 
          {
            sensorCheckState = SENSOR_CHECK_STATE_NORMAL;
          }
        break;

        case SENSOR_CHECK_STATE_NORMAL:
          if(controlSensorAdcCount < LOAD_CELL_INPUT_SHORTED_VALUE)
          {
            Serial.println("\nLoad cell input too low. May be shorted!");
            sprintf(gPrintBuf, "CS runtime value = %d", controlSensorAdcCount);
            Serial.println(gPrintBuf);
            stopAirFlow();
            timerStartValue = millis();
            sensorCheckState = SENSOR_CHECK_STATE_OUT_OF_RANGE;
          }
          else if(controlSensorAdcCount > LOAD_CELL_INPUT_OPEN_VALUE)
          {
            Serial.println("\nLoad cell input too high. May be open!");
            sprintf(gPrintBuf, "CS runtime value = %d", controlSensorAdcCount);
            Serial.println(gPrintBuf);
            stopAirFlow();
            timerStartValue = millis();
            sensorCheckState = SENSOR_CHECK_STATE_OUT_OF_RANGE;
          }
          else if(forceSensorAdcCount < FORCE_SENSOR_INPUT_SHORTED_VALUE)
          {
            Serial.println("\nForce sensor input too low. May be shorted!");
            sprintf(gPrintBuf, "FS runtime value = %d", forceSensorAdcCount);
            Serial.println(gPrintBuf);
            stopAirFlow();
            timerStartValue = millis();
            sensorCheckState = SENSOR_CHECK_STATE_OUT_OF_RANGE;
          }
          else if(forceSensorAdcCount > FORCE_SENSOR_INPUT_OPEN_VALUE)
          {
            Serial.println("\nForce sensor input too high. May be open!");
            sprintf(gPrintBuf, "FS runtime value = %d", forceSensorAdcCount);
            Serial.println(gPrintBuf);
            stopAirFlow();
            timerStartValue = millis();
            sensorCheckState = SENSOR_CHECK_STATE_OUT_OF_RANGE;
          }
          else 
          {
          }                    
        break;
        
        case SENSOR_CHECK_STATE_OUT_OF_RANGE:
            //blink RED LED. Power cycle required
            if((millis() - timerStartValue) > 100)
            {
              timerStartValue = millis();
              digitalWrite(LED_ERROR, ledToggle); 
              ledToggle = ledToggle ^ 1;
            }
        break;
        default:
          Serial.println("\nHit default in check sensor limits");
        break;
    }
    return sensorCheckState;
}