#include <arduino.h>
#include <EEPROM.h>
#include "zg_balancer.h"
#include "zg_capacity.h"
extern char gPrintBuf[50];
int gTmpTotalTimeLapsed = 0;
static uint8_t ledToggle = 1;

uint8_t runHandleMode(const stSensorData *pstForceSensorData, stSensorData *pstControlSensorData, const stZgSettings *pStZgSettings)
{
  static uint8_t handleModeState = HANDLE_MODE_STATE_IDLE;
  static unsigned long timerStartValue = 0;
  int tmp;
  //static uint16_t prevReading = 0;
  uint8_t weightRange;

  switch (handleModeState)
  {
    case HANDLE_MODE_STATE_IDLE:
      if (TREND_STATE_INCREASED == pstForceSensorData->trend)
      {
        handleModeState = HANDLE_MODE_STATE_DOWN;
      }
      else if (TREND_STATE_DECREASED == pstForceSensorData->trend)
      {
        handleModeState = HANDLE_MODE_STATE_UP;
      }
      else
      {
        handleModeState = HANDLE_MODE_STATE_IDLE;
      }
      break;
    case HANDLE_MODE_STATE_UP:
      //user is trying to pull the load up using handle
      weightRange = getCurrentWeightRange(pstControlSensorData->rawAdcCounts);
      airIntake(pStZgSettings->liftSpeed, weightRange);
      //operate yellow LEDs when interlock is not set
      if (0 == pStZgSettings->interlockWeight)
      {
        digitalWrite(LED1, LOW);
        digitalWrite(LED2, LOW);
        digitalWrite(LED3, LOW);
      }
      handleModeState = HANDLE_MODE_STATE_UP_IN_PROGRESS;
      Serial.println("\nHANDLE_MODE_UP_IN_PROGRESS...");
      break;
    case HANDLE_MODE_STATE_UP_IN_PROGRESS:
      //wait for the user to release the handle after lifting
      if (TREND_STATE_DECREASED == pstForceSensorData->trend)
      {
        //user has not released the handle
        weightRange = getCurrentWeightRange(pstControlSensorData->rawAdcCounts);
        airIntake(pStZgSettings->liftSpeed, weightRange);
        
        //buffer for max weight
        if(pstControlSensorData->rawAdcCounts > pStZgSettings->maxWeightToLift + (ONE_KG_WEIGHT * 10))
        {
          //allowing this margin to allow abuse by user (rapid upward handle movement)
          handleModeState = HANDLE_MODE_STATE_HIT_MAX_WEIGHT;
          stopAirFlow();
          timerStartValue = millis();
        }
              
      }
      else if (TREND_STATE_INCREASED == pstForceSensorData->trend)
      {
        handleModeState = HANDLE_MODE_STATE_DOWN;
      }
      else
      {
        stopAirFlow();
        timerStartValue = millis();
        gTmpTotalTimeLapsed = 0;
        digitalWrite(LED_ERROR, HIGH);  //indicate we are busy
        handleModeState = HANDLE_MODE_STATE_WAIT_FOR_CONTROL_SENSOR_RESTING;
        Serial.println("\nWAIT_FOR_CONTROL_SENSOR_RESTING...");
      }
      break;

    case HANDLE_MODE_STATE_DOWN:
      //user is trying to pull the load down using handle
      weightRange = getCurrentWeightRange(pstControlSensorData->rawAdcCounts);
      airExhaust(pStZgSettings->lowerSpeed, weightRange);
      //operate yellow LEDs when interlock is not set      
      if (0 == pStZgSettings->interlockWeight)
      {
        digitalWrite(LED1, LOW);
        digitalWrite(LED2, LOW);
        digitalWrite(LED3, LOW);
      }

      handleModeState = HANDLE_MODE_STATE_DOWN_IN_PROGRESS;
      Serial.println("\nHANDLE_MODE_DOWN_IN_PROGRESS...");

      break;
    case HANDLE_MODE_STATE_DOWN_IN_PROGRESS:
      //keep exhausting air to decrease pressure until handle position is restored to original position
      if (TREND_STATE_INCREASED == pstForceSensorData->trend)
      {
        //user has not released the handle
        weightRange = getCurrentWeightRange(pstControlSensorData->rawAdcCounts);
        airExhaust(pStZgSettings->lowerSpeed, weightRange);
        //check for min weight or no load balancing
        if(pstControlSensorData->rawAdcCounts < pStZgSettings->noLoadBalancingWeight)
        {
          handleModeState = HANDLE_MODE_STATE_HIT_MIN_WEIGHT;
          stopAirFlow();
          timerStartValue = millis();
        }      
        
      }
      else if (TREND_STATE_DECREASED == pstForceSensorData->trend)
      {
        handleModeState = HANDLE_MODE_STATE_UP;
      }
      else
      {
        stopAirFlow();
        timerStartValue = millis();
        gTmpTotalTimeLapsed = 0;
        digitalWrite(LED_ERROR, HIGH);  //indicate we are busy
        handleModeState = HANDLE_MODE_STATE_WAIT_FOR_CONTROL_SENSOR_RESTING;
        Serial.println("\nWAIT_FOR_CONTROL_SENSOR_RESTING...");
      }

      break;
    case HANDLE_MODE_STATE_WAIT_FOR_CONTROL_SENSOR_RESTING:
      //always honour handle operation while checking for load cell to settle
      //wait for the load cell to settle down. Else we will be unduly triggering
      //float mode

      //sprintf(gPrintBuf, "diff counts = %d", gTmpTotalTimeLapsed);
      //Serial.println(gPrintBuf);

      if (TREND_STATE_INCREASED == pstForceSensorData->trend)
      {
        handleModeState = HANDLE_MODE_STATE_DOWN;
      }
      else if (TREND_STATE_DECREASED == pstForceSensorData->trend)
      {
        handleModeState = HANDLE_MODE_STATE_UP;
      }
      //find the new resting value (we may be lifting load from ground)
      if((millis() - timerStartValue) > 5)
      {
        //every 5 ms check if are coming to rest
        if (pstControlSensorData->trend != TREND_STATE_RESTING)
        {
          tmp = millis() - timerStartValue;
          gTmpTotalTimeLapsed += tmp;
          timerStartValue = millis();
          //Serial.println(pstControlSensorData->trend);
          //Serial.println(pstControlSensorData->delta);
          //adjust resting value
          pstControlSensorData->restingValue = pstControlSensorData->rawAdcCounts;
        }
        else if(millis() - timerStartValue > CONTROL_SENSOR_RESTING_TIMER_MS)
        {
          tmp = millis() - timerStartValue;
          gTmpTotalTimeLapsed += tmp;
          digitalWrite(LED_ERROR, LOW);
          //light up the yellow LEDs to indicate float mode ready
          //if interlock is not selected
          if (0 == pStZgSettings->interlockWeight)
          {
            digitalWrite(LED1, HIGH);
            digitalWrite(LED2, HIGH);
            digitalWrite(LED3, HIGH);          
          }
          pstControlSensorData->restingValue = pstControlSensorData->filteredAdcCounts;
          handleModeState = HANDLE_MODE_STATE_IDLE;
          sprintf(gPrintBuf, "Time for load cell settling = %d ms", gTmpTotalTimeLapsed);
          Serial.println(gPrintBuf);
          Serial.println("\nHANDLE_MODE_DOWN_IDLE");
        }
      }
      break;
    case HANDLE_MODE_STATE_HIT_MAX_WEIGHT:
      //blink RED LED
      if((millis() - timerStartValue) > 250)
      {
        timerStartValue = millis();
        digitalWrite(LED_ERROR, ledToggle); 
        ledToggle = ledToggle ^ 1;
      }
      //allow only handle down operation
      if (TREND_STATE_INCREASED == pstForceSensorData->trend)
      {
        handleModeState = HANDLE_MODE_STATE_DOWN;
        digitalWrite(LED_ERROR, 0); 
      }
      //in case settings are changed while we are in this state
      // if(pstControlSensorData->rawAdcCounts < pStZgSettings->maxWeightToLift)
      // {
      //   digitalWrite(LED_ERROR, 0); 
      //   handleModeState = HANDLE_MODE_STATE_IDLE;        
      // }
      break;
    case HANDLE_MODE_STATE_HIT_MIN_WEIGHT:
      //blink RED LED
      if((millis() - timerStartValue) > 250)
      {
        timerStartValue = millis();
        digitalWrite(LED_ERROR, ledToggle); 
        ledToggle = ledToggle ^ 1;
      }
      //allow only handle up operation
      if (TREND_STATE_DECREASED == pstForceSensorData->trend)
      {
        handleModeState = HANDLE_MODE_STATE_UP;
        digitalWrite(LED_ERROR, 0); 
      }
      // //in case settings are changed while we are in this state
      // if(pstControlSensorData->rawAdcCounts >= pStZgSettings->noLoadBalancingWeight)
      // {
      //   digitalWrite(LED_ERROR, 0); 
      //   handleModeState = HANDLE_MODE_STATE_IDLE;        
      // }
      break;
      
    default:
      Serial.println("Hit default case in run handle mode");
      break;
  }

  return handleModeState;
}
