#include <stdint.h>
#ifndef ZG_BALANCER_H
#define ZG_BALANCER_H

#define SW_VERSION 0x01010001  //4th byte = major, 3rd byte = minor, 2nd and 1st bytes = build number

#define MOVING_WINDOW_SIZE 10
#define SERIAL_NUMBER_LEN 14

#define EXHAUST_LOW_SOLENOID 9
#define EXHAUST_MED_SOLENOID 8
#define EXHAUST_HIGH_SOLENOID 7

#define INTAKE_LOW_SOLENOID 12
#define INTAKE_MED_SOLENOID 11
#define INTAKE_HIGH_SOLENOID 10

#define DI_ENGAGE_CLAMPING A1    /*used as digital inputs */
#define DI_DISENGAGE_CLAMPING A2 /*used as digital inputs */

#define LED1 6
#define LED2 5
#define LED3 4
#define LED_ERROR 3

#define SOLENOID_OPEN 1
#define SOLENOID_CLOSE 0

//these limits needs to be reviewed
// #define FORCE_SENSOR_INIT_CHECK_HIGH_LIMIT 550
// #define FORCE_SENSOR_INIT_CHECK_LOW_LIMIT 450
// #define FORCE_SENSOR_RUN_TIME_CHECK_HIGH_LIMIT 1000
// #define FORCE_SENSOR_RUN_TIME_CHECK_LOW_LIMIT 10

//#define CONTROL_SENSOR_INIT_CHECK_HIGH_LIMIT 750

#define CONTROL_SENSOR_RESTING_TIMER_MS 500 /* control sensor should continuously idle this much time before we decide the new resting value */


//which way the sensor reading are moving
#define TREND_STATE_INIT 0
#define TREND_STATE_RESTING_CHECK 1
#define TREND_STATE_RESTING 2
#define TREND_STATE_INCREASED 3
#define TREND_STATE_DECREASED 4
#define HANDLE_MODE_OVERRIDE_CLEAR 5
#define HANDLE_MODE_OVERRIDE_UP 6
#define HANDLE_MODE_OVERRIDE_DOWN 7

//balancer state machine states
#define HANDLE_MODE_STATE_IDLE 0
#define HANDLE_MODE_STATE_UP 1
#define HANDLE_MODE_STATE_DOWN 2
#define HANDLE_MODE_STATE_UP_IN_PROGRESS 3
#define HANDLE_MODE_STATE_DOWN_IN_PROGRESS 4
#define HANDLE_MODE_STATE_WAIT_FOR_CONTROL_SENSOR_RESTING 5
#define HANDLE_MODE_STATE_HIT_MAX_WEIGHT 6
#define HANDLE_MODE_STATE_HIT_MIN_WEIGHT 7

#define FLOAT_MODE_STATE_IDLE 0
#define FLOAT_MODE_STATE_UP 1
#define FLOAT_MODE_STATE_UP_IN_PROGRESS 2
#define FLOAT_MODE_STATE_DOWN 3
#define FLOAT_MODE_STATE_DOWN_IN_PROGRESS 4
#define FLOAT_MODE_STATE_HIT_RAPID_WEIGHT_CHANGE 5
#define FLOAT_MODE_STATE_HIT_MIN_WEIGHT 6



//minimum ADC count change to consider for determining user action
//these are derived during testing on the balancer may be used for sensitivity control
//these limits are with respect to resting value (adc reading when things are idling)
#define FORCE_SENSOR_CONTROL_DEAD_BAND_HI_LIMIT 50
#define FORCE_SENSOR_CONTROL_DEAD_BAND_LO_LIMIT 50
#define FORCE_SENSOR_RESTING_DEAD_BAND_HI_LIMIT 20
#define FORCE_SENSOR_RESTING_DEAD_BAND_LO_LIMIT 20

#define AI_FORCE_SENSOR A0
#define AI_LOAD_CELL A3
#define AI_NO_LOAD_BALANCING_POT A11
#define AI_INTERLOCK_POT A12
#define AI_MAX_PRESSURE_POT A13
#define AI_PRESSURE_SENSOR A15

#define EEPROM_ADDR_START 0
//settings...
#define EEPROM_ADDR_SETTINGS_START EEPROM_ADDR_START
#define NUM_ZG_SETTINGS_BYTES  sizeof(stZgSettings)
#define EEPROM_ADDR_SETTINGS_END ((EEPROM_ADDR_SETTINGS_START) + NUM_ZG_SETTINGS_BYTES)

//persistent data
#define EEPROM_ADDR_ZG_DATA_START ((EEPROM_ADDR_SETTINGS_END) + 1)
#define NUM_ZG_DATA_BYTES sizeof(stZgData)
#define EEPROM_ADDR_ZG_DATA_END ((EEPROM_ADDR_ZG_DATA_START) + NUM_ZG_DATA_BYTES)

//speed settings
#define LOW_SPEED 0
#define MED_SPEED 1
#define HIGH_SPEED 2
#define MAX_SPEED_RANGES 7





//err codes
#define ERR_SENSORS_OUT_OF_RANGE 0

//serial protocol
#define START_BYTE 0xAA
#define MULTIPLE_READ_CMD 0x01
#define MULTIPLE_WRITE_CMD 0x02
#define HANDLE_MODE_OVERRIDE_CMD 0x03
#define KEEP_ALIVE_CMD 0x04
#define SET_SERIAL_NUM_CMD 0x05
#define GET_SERIAL_NUM_CMD 0x06

#define NACK_RESPONSE 0x00
#define MULTIPLE_READ_RESPONSE 0x81
#define MULTIPLE_WRITE_RESPONSE 0x82
#define HANDLE_MODE_OVERRIDE_RESPONSE 0x83
#define KEEP_ALIVE_RESPONSE 0x84
#define SET_SERIAL_NUM_RESPONSE 0x85
#define GET_SERIAL_NUM_RESPONSE 0x86

#define NUM_PING_BYTES 10
#define PING_RESPONSE_LEN 13
#define GET_SERIAL_NUMBER_RESPONSE_LEN 17
#define SET_SERIAL_NUMBER_RESPONSE_LEN 17
#define HANDLE_MODE_RESPONSE_LEN 4

#define HM_STATE_NONE 1
#define HM_STATE_TIME_CHECK 2
#define HANDLE_MODE_TIMEOUT_MS 500

#define RX_STATE_WAIT_FOR_START_BYTE 1
#define RX_STATE_WAIT_FOR_LEN_BYTE 2
#define RX_STATE_WAIT_FOR_PAYLOAD 3

//parameter ids
//1 to 30 reserved for setting parameters
#define PARAM_ID_SW_VERSION 1
#define PARAM_ID_LOAD_CELL_DEAD_BAND_HI_LIMIT 2
#define PARAM_ID_LOAD_CELL_DEAD_BAND_LO_LIMIT 3
#define PARAM_ID_LIFT_SPEED 4
#define PARAM_ID_LOWER_SPEED 5
#define PARAM_ID_INTERLOCK_WEIGHT 6
#define PARAM_ID_MAX_WEIGHT 7
#define PARAM_ID_MIN_WEIGHT 8

//live data paramter ids
#define PARAM_ID_LOAD_CELL_SENSOR 31
#define PARAM_ID_FORCE_SENSOR 32
#define PARAM_ID_PRESSURE_SENSOR 33
#define PARAM_ID_SETTLING_TIME 34
#define PARAM_ID_CURRENT_WEIGHT_RANGE 35
#define PARAM_ID_CYCLE_COUNT 36
#define PARAM_ID_UN_USED37 37

#define MAX_PARAM_IDS 15

#define CYCLE_COUNT_STATE_INIT 0
#define CYCLE_COUNT_STATE_AIR_LIFTED 1
#define CYCLE_COUNT_STATE_GROUNDED 2

#define SENSOR_CHECK_STATE_CHECK_RESTING_VALUES  1
#define SENSOR_CHECK_STATE_NORMAL                2
#define SENSOR_CHECK_STATE_OUT_OF_RANGE          3

#define LOAD_CELL_INPUT_SHORTED_VALUE            20
//adc count > 90% capacity ?
//#define LOAD_CELL_INPUT_OPEN_VALUE               int(ZG_CAPACITY * 0.453592 * ONE_KG_WEIGHT * 0.9)
//#define LOAD_CELL_INPUT_OPEN_VALUE               int(ZG_CAPACITY * 0.453592 * ONE_KG_WEIGHT)
#define LOAD_CELL_INPUT_OPEN_VALUE               1010
#define FORCE_SENSOR_INPUT_SHORTED_VALUE         35
#define FORCE_SENSOR_INPUT_OPEN_VALUE            990
#define FORCE_SENSOR_INPUT_RESTING_LOW_LIMIT     500 //ideally (512 - 20) adc count
#define FORCE_SENSOR_INPUT_RESTING_HIGH_LIMIT    750 //ideally (512 + 20) adc counts

//#define HANDLE_WEIGHT 20  //(without any weight on the hook, we get appr 15 ADC counts)
#define SETTINGS_SAVE_TIME_INTERVAL 300000
//#define PRESSURE_SENSOR_CONTROL
#define LOAD_CELL_SENSOR_CONTROL

#ifdef PRESSURE_SENSOR_CONTROL
#define AI_CONTROL_SENSOR AI_PRESSURE_SENSOR
#else
#ifdef LOAD_CELL_SENSOR_CONTROL
#define AI_CONTROL_SENSOR AI_LOAD_CELL
#else
#error "Control sensor not defined"
#endif
#endif

typedef struct sensorData
{
  uint8_t isControlSensor;
  uint8_t trend;
  uint8_t trendManualOverride;
  //deadbands are wrt resting value
  //Take action if values are outside ctrl deadband
  uint8_t controlDeadbandHiLImit;
  uint8_t controlDeadbandLoLImit;
  //sensor is considered resting when its values
  //is within resting deadband
  uint8_t restingDeadbandHiLimit;
  uint8_t restingDeadbandLoLimit;
  uint16_t rawAdcCounts;
  uint16_t filteredAdcCounts;
  uint16_t restingValue;
  int16_t delta;  //with respect to resting value
  float oldFilteredReading;
} stSensorData;

typedef struct zgSettings
{
  uint8_t liftSpeed;
  uint8_t lowerSpeed;
  uint16_t interlockWeight;
  uint16_t maxWeightToLift;  
  uint16_t noLoadBalancingWeight; 
  //deadband for float mode and handle mode (or sensitivity control)
  uint8_t floatModeControlDeadbandHiLImit;
  uint8_t floatModeControlDeadbandLoLImit;
  uint8_t floatModeRestingDeadbandHiLImit;
  uint8_t floatModeRestingDeadbandLoLImit;

  uint8_t handleModeControlDeadbandHiLImit;
  uint8_t handleModeControlDeadbandLoLImit;
  uint8_t handleModeRestingDeadbandHiLImit;
  uint8_t handleModeRestingDeadbandLoLImit;
} stZgSettings;

typedef struct zgData
{
  unsigned long cycleCount;
  char serialNumber[SERIAL_NUMBER_LEN];
  //add future data points here
} stZgData;

typedef struct wtVsSpeed
{
  uint8_t mValveLiftingCombination;
  uint8_t dValveLiftingCombination;
  uint8_t mValveLoweringCombination;
  uint8_t dValveLoweringCombination;
} stWtRangeVsSpeedLookUp;

typedef union word2Bytes
{
  uint8_t byteArr[4];
  unsigned long fourByteValue;
  uint16_t twoByteValue;
} unWord2Bytes;

void startPressureIncrese(int speedSetting);
void startPressureDecrease(int speedSetting);
void stopAirFlow();
void initInterlockInputs();
void initLedOutputs();
void initExhaustSolenoids();
void initIntakeSolenoids();
void monitorSensors();

void safeExit(int errCode);

int adcRead(int ch);
int adcReadWithFilter(int ch);
/*int getMovingAvg(int currentValue, stMovingAvgData *stAvgData);*/
int getSensorRestingValue(int8_t adcCh, uint8_t minReadingChange);
//int getSensorTrend(int currentAvg, stTrendData *stSensorTrendData);
int getAvgSensorReading(int sensor, int numSamplesAvg);
int getBalancerState(int forceSensorTrend, int pressureSensorTrend);
uint8_t getCurrentWeightRange(uint16_t controlSensorAdcCounts);
void createWeightVsSpeedLookUpTable();
void initSettings(stZgSettings *pStZgSettings);
//void saveSettings(stZgSettings *pStZgSettings);
void initZgData(stZgData *pStZgData);
//void saveZgData(stZgData *pStZgData);
void saveZgDataPeriodically(stZgData *pStCurrentZgData);
void WriteToEeprom(int32_t eepromAddr, uint8_t *dataBuf, int16_t numBytesToWrite);
void airIntake(uint8_t speedSetting, uint8_t weightRange);
void airExhaust(uint8_t speedSetting, uint8_t weightRange);
void handleCycleCount(const stSensorData *pstControlSensorData, stZgData *pstZgData, stZgSettings *pStZgSettings);
void handleInterlock(const stSensorData *, const stZgSettings *);
uint8_t getSerialCmd(unsigned char *reqPayLoadBuf);
uint8_t processPayload(unsigned char *reqPayLoadBuf, stSensorData *pstForceSensorData, stSensorData *pstControlSensorData, stZgSettings *pStZgSettings, stZgData *pstZgData);
//uint8_t processPayload(unsigned char *reqPayLoadBuf);
void checkHandleModeOverride(uint8_t cmdReceived, stSensorData *pstForceSensorData);

int verifyChecksum(int payLoadLen, unsigned char *payLoadBuf);
uint8_t runHandleMode(const stSensorData *pstForceSensorData, stSensorData *pstControlSensorData, const stZgSettings *pStZgSettings);
uint8_t runFloatMode(stSensorData *pstControlSensorData, const stZgSettings *pStZgSettings);
void initSensorData(stSensorData *pstSensorData, uint8_t ch, stZgSettings *pstZgSettings);
void updateSensorData(stSensorData *pstSensorData, uint16_t adcCounts);
uint8_t checkSensors(uint16_t controlSensorAdcCount, uint16_t forceSensorAdcCount);
//uint8_t setParamValue(uint8_t paramId, unsigned long paramValue);
//unsigned long getParamValue(uint8_t paramId);
#endif
